/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	$Id: PGPPassphraseDialog.cpp,v 1.2 2001/03/21 00:12:53 dallen Exp $
____________________________________________________________________________*/

#include <gnome.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "pgpDialogs.h"
#include "pgpPassphraseUtils.h"
#include "pgpErrors.h"
#include "pgpKeys.h"
#include "pgpMem.h"
#include "PGPuiInterface.h"
#include "PGPuiGladeSupport.h"
#include "PGPSpaceEdit.h"

#define MAXDECRYPTIONNAMECHAR		36

// global variable structure for re-entrancy
typedef struct _GPP
{
	char *				pszPassPhrase;
	char *				pszPassPhraseConf;
	PGPInt32			iNextTabControl;
	PGPBoolean			bHideText;
	PGPContextRef		context;
	const CPGPPassphraseDialogOptions *options;
} GPP;

PGPError
PGPsdkUIErrorBox (
		PGPError error) 
{
	PGPError	err				= kPGPError_NoErr;
	char		szMessage[512];

	if (IsPGPError (error) && (error!=kPGPError_UserAbort)) 
	{
		PGPGetErrorString (error, sizeof(szMessage), szMessage);

		printf("%s: PGP Error", szMessage);
	}

	return err;
}

	PGPUInt32
PGPEstimatePassphraseQuality(const char *passphrase)
{
	return( pgpEstimatePassphraseQuality( passphrase ) );
}

//___________________________
//
// Secure memory allocation routines
//

void * 
secAlloc (PGPContextRef context, PGPUInt32 uBytes) 
{
	PGPMemoryMgrRef	memmgr;

	memmgr = PGPPeekContextMemoryMgr (context);
	return (PGPNewSecureData (memmgr, uBytes, 0));
}


void 
secFree (void* p) 
{
	if (p) {
		memset ((char *)p, '\0', strlen((char *)p));
		PGPFreeData ((char *)p);
	}
}

//	__________________
//
//	Wipe edit box clean

void 
WipeEditBox (
		GPP *gpp,
		PGPUInt32 uID) 
{
	(void) gpp;
	(void) uID;
}

void FreePassphrases(GPP *gpp)
{
	if (gpp->pszPassPhrase) {
		secFree(gpp->pszPassPhrase);
		gpp->pszPassPhrase=NULL;
	}

	if (gpp->pszPassPhraseConf) {
		secFree(gpp->pszPassPhraseConf);
		gpp->pszPassPhraseConf=NULL;
	}

}

void ClearPassphrases(GPP *gpp)
{
	if(gpp->pszPassPhraseConf) {
		secFree(gpp->pszPassPhraseConf);
		gpp->pszPassPhraseConf=NULL;
	}

}

//	____________________________
//
//  setup key display list box

PGPBoolean 
AddKeySetToRecipientsTable (PGPKeySetRef KeySet,
							CPGPDecryptionPassphraseDialogOptions *options) 
{
	PGPBoolean			bAtLeastOneValidSecretKey	= FALSE;
	PGPKeyListRef	KeyList;
	PGPKeyIterRef	KeyIter;
	PGPKeyDBObjRef	Key;
	PGPKeyDBObjRef	SubKey;
	PGPSize			u;
	PGPUInt32			uKeyBits;
	PGPUInt32			uAlgorithm;
	PGPInt32				iKeyDefault, iKeySelected;
	PGPBoolean		bSecret, bCanEncrypt;
	char			szName[kPGPMaxUserIDSize];
	char			sz[128];

	(void) options;
	PGPOrderKeySet (KeySet, kPGPKeyOrdering_Validity, FALSE, &KeyList);
	PGPNewKeyIter (KeyList, &KeyIter);

	iKeySelected = -1;
	iKeyDefault = -1;

	PGPKeyIterNextKeyDBObj (KeyIter, kPGPKeyDBObjType_Key, &Key);
	while (Key) 
	{
		PGPGetKeyDBObjBooleanProperty (Key, kPGPKeyProperty_IsSecret, &bSecret);
		PGPGetKeyDBObjBooleanProperty (Key, kPGPKeyProperty_CanEncrypt, &bCanEncrypt);
		PGPGetKeyDBObjNumericProperty (Key, kPGPKeyProperty_AlgorithmID,
			(int *)&uAlgorithm);
		if (bSecret && bCanEncrypt) bAtLeastOneValidSecretKey = TRUE;

		// get name on key
		PGPGetPrimaryUserIDName (Key, szName, sizeof(szName), &u);
		if (u > (PGPSize)MAXDECRYPTIONNAMECHAR) 
		{
			u = (PGPSize)MAXDECRYPTIONNAMECHAR;
			strcat (szName, "...");
		}
		else 
			szName[u] = '\0';

		// append key type / size info to name
		strcat (szName, " (");

		switch (uAlgorithm) 
		{
			case kPGPPublicKeyAlgorithm_RSA :
				strcat (szName, sz);
				strcat (szName, "/");
				PGPGetKeyDBObjNumericProperty (Key, kPGPKeyProperty_Bits,
					(int *)&uKeyBits);
				sprintf (sz, "%i", uKeyBits);
				strcat (szName, sz);
				break;

			case kPGPPublicKeyAlgorithm_DSA :
				strcat (szName, sz);
				strcat (szName, "/");
				PGPKeyIterNextKeyDBObj (KeyIter,
					 kPGPKeyDBObjType_SubKey,&SubKey);
				if (SubKey) {
					PGPGetKeyDBObjNumericProperty (SubKey, kPGPKeyProperty_Bits, 
										(int *)&uKeyBits);
					sprintf (sz, "%i", uKeyBits);
					strcat (szName, sz);
				}
				else strcat (szName, "???");
				break;

			default :
				strcat (szName, sz);
				strcat (szName, "/");
				strcat (szName, sz);
				break;
		}
		strcat (szName, ")");

		PGPKeyIterNextKeyDBObj (KeyIter, kPGPKeyDBObjType_Key, &Key);
	}
	PGPFreeKeyIter (KeyIter);
	PGPFreeKeyList (KeyList);

	return bAtLeastOneValidSecretKey;
}

PGPBoolean 
InitEncryptedToKeyListBox (CPGPDecryptionPassphraseDialogOptions *options) 
{
	PGPBoolean		bAtLeastOneValidSecretKey	= FALSE;
	PGPKeySetRef	keySet						= kInvalidPGPKeySetRef;

	if(PGPKeySetRefIsValid( options->mKeySet ))
		bAtLeastOneValidSecretKey = AddKeySetToRecipientsTable(
										options->mKeySet,options );

	if(PGPKeyDBRefIsValid( *(options->mNewKeys) ))
		PGPNewKeySet (*(options->mNewKeys), &keySet);
	if(PGPKeySetRefIsValid(keySet))
		AddKeySetToRecipientsTable(keySet,options );

	if( options->mMissingKeyIDList != NULL )
	{
		char MsgTxt[255];
	
		sprintf(MsgTxt,"%d unknown key(s)",options->mMissingKeyIDCount);

	}

	return bAtLeastOneValidSecretKey;
}

static
void GetKeyString(PGPKeyDBObjRef Key,char *szNameFinal)
{
	char			sz1[32],sz2[32];
	char			szName[kPGPMaxUserIDSize];
	PGPUInt32		uAlgorithm,uKeyBits;
	PGPSize			u;

	PGPGetKeyDBObjNumericProperty (Key, kPGPKeyProperty_AlgorithmID,
		(int *)&uAlgorithm);

	// get key type / size info to append to name
	strcpy (sz2, "   (");
	switch (uAlgorithm) 
	{
	case kPGPPublicKeyAlgorithm_RSA :
		strcat (sz2, "RSA/");
		PGPGetKeyDBObjNumericProperty (Key, kPGPKeyProperty_Bits, (int *)&uKeyBits);
		sprintf (sz1, "%i", uKeyBits);
		strcat (sz2, sz1);
		break;
		
	case kPGPPublicKeyAlgorithm_DSA :
		strcat (sz2, "DSS/");
		PGPGetKeyDBObjNumericProperty (Key, kPGPKeyProperty_Bits, (int *)&uKeyBits);
		sprintf (sz1, "%i", uKeyBits);
		strcat (sz2, sz1);
		break;
		
	default :
		strcat (sz2, "Unknown/Unknown");
		break;
	}
	
	strcat (sz2, ")");

	// get name on key
	PGPGetPrimaryUserIDName (Key, szName, sizeof(szName), &u);

	strcpy(szNameFinal, szName);
	strcat(szNameFinal, sz2);
	
	//TruncateKeyText (hdc, szName, sz2, iComboWidth, szNameFinal);
}


//	____________________________
//
//  setup keyselection list o' keys
//

PGPBoolean 
InitSigningKeyComboBox (CPGPKeySetPassphraseDialogOptions *options) 
{
	PGPKeyListRef	KeyList;
	PGPKeyIterRef	KeyIter;
	PGPKeyDBObjRef	Key;
	PGPKeyDBObjRef	KeyDefault	= kInvalidPGPKeyDBObjRef;
	PGPUInt32		uIndex;
	PGPInt32		iKeyDefault, iKeySelected;
	PGPBoolean		bSecret, bRevoked, bExpired, bCanSign;
	PGPBoolean		bAtLeastOneSecretKey;
	char			szNameFinal[kPGPMaxUserIDSize];

	PGPOrderKeySet (options->mKeySet, kPGPKeyOrdering_Validity, FALSE, &KeyList);
	PGPNewKeyIter (KeyList, &KeyIter);
//	PGPGetDefaultPrivateKey (options->mKeySet, &KeyDefault);

	iKeySelected = -1;
	iKeyDefault = 0;

	bAtLeastOneSecretKey = FALSE;

	PGPKeyIterNextKeyDBObj (KeyIter, kPGPKeyDBObjType_Key, &Key);
	while (Key) {
		PGPGetKeyDBObjBooleanProperty (Key, kPGPKeyProperty_IsSecret, &bSecret);
		if (bSecret) {
			PGPGetKeyDBObjBooleanProperty (Key, kPGPKeyProperty_IsRevoked,
							  (unsigned char *)&bRevoked);
			PGPGetKeyDBObjBooleanProperty (Key, kPGPKeyProperty_IsExpired,
							  (unsigned char *)&bExpired);
			PGPGetKeyDBObjBooleanProperty (Key, kPGPKeyProperty_CanSign,
							  (unsigned char *)&bCanSign);
			if (!bRevoked && !bExpired && bCanSign) {
				bAtLeastOneSecretKey = TRUE;

				GetKeyString(Key,szNameFinal);

				fprintf(stdout, "\n%s\n", szNameFinal);
				if (options->mDefaultKey) {
					if (Key == options->mDefaultKey) 
						iKeySelected = uIndex;
				}
				if (Key == KeyDefault) iKeyDefault = uIndex;
			}
		}
		PGPKeyIterNextKeyDBObj (KeyIter, kPGPKeyDBObjType_Key, &Key);
	}
	PGPFreeKeyIter (KeyIter);
	PGPFreeKeyList (KeyList);

	return (bAtLeastOneSecretKey);

}

PGPBoolean PassphraseLengthAndQualityOK(CPGPPassphraseDialogOptions *options,
									char *Passphrase)
{
	if(options->mMinPassphraseLength!=0)
	{
		if(strlen(Passphrase)<options->mMinPassphraseLength)
		{
			printf("Passphrase is not of sufficient length. "
						"Please choose another.");
			return FALSE;
		}
	}

	if(options->mMinPassphraseQuality!=0)
	{
		if(PGPEstimatePassphraseQuality(Passphrase) <
				options->mMinPassphraseQuality)
		{
			printf("Passphrase is not of sufficient quality. "
						"Please choose another.");
			return FALSE;
		}
	}

	return TRUE;
}

//	____________________________
//
//  search keys for matching phrase

PGPError 
ValidateSigningPhrase (GPP *gpp, char * pszPhrase, PGPKeyDBObjRef key) 
{
	char	szName[kPGPMaxUserIDSize];
	char	sz[128];
	char	sz2[kPGPMaxUserIDSize + 128];
	PGPSize	size;
	CPGPSigningPassphraseDialogOptions *options;

	options = (CPGPSigningPassphraseDialogOptions *)gpp->options;

	// does phrase match selected key ?
	if (PGPPassphraseIsValid (key, 
			PGPOPassphrase (gpp->context, pszPhrase),
			PGPOLastOption (gpp->context))) {
		*(options->mPassphraseKeyPtr) = key;
		return kPGPError_NoErr;
	}

	if (options->mFindMatchingKey) {
		// does phrase match any private key ?
		key=GetKeyForPassphrase(options->mKeySet,pszPhrase,TRUE);

		if (key != NULL) {
			// ask user to use other key
			PGPGetPrimaryUserIDName (key, szName, sizeof(szName), &size);
			sprintf (sz2, sz, szName);
			return kPGPError_BadPassphrase;
		}
	}

	// phrase doesn't match any key
	printf("Bad Passphrase: Please re-enter\n");
	
	return kPGPError_BadPassphrase;

}

// ****************************************************************************
// ****************************************************************************

// Just a simple decryption
	PGPError
pgpPassphraseDialogPlatform(
	PGPContextRef					context,
	CPGPPassphraseDialogOptions 	*options)
{
	PGPError err = 0;
	GPP	gpp;

	memset(&gpp,0x00,sizeof(GPP));
	gpp.context=context;
	gpp.options=options;

//	InitRandomKeyHook(&hhookMouse,&hhookKeyboard);

//	err = DialogBoxParam (g_hInst, 
	//	MAKEPGPInt32 (IDD_PASSPHRASE), 
	//	options->mHwndParent,
	//	(DLGPROC)pgpPassphraseDlgProc, (LPARAM)&gpp);

//	*(options->mPassphrasePtr)=gpp.pszPassPhrase;

//	UninitRandomKeyHook(hhookMouse,hhookKeyboard);

	return(err);
}

// Show the recipients
	PGPError
pgpDecryptionPassphraseDialogPlatform(
	PGPContextRef							context,
	CPGPDecryptionPassphraseDialogOptions	*options)
{
	PGPError err = 0;
	GPP	gpp;

	memset(&gpp,0x00,sizeof(GPP));
	gpp.context=context;
	gpp.options=options;

//	InitRandomKeyHook(&hhookMouse,&hhookKeyboard);

	//err = DialogBoxParam (g_hInst, 
	//	MAKEINTRESOURCE (IDD_PASSPHRASEDECRYPTKEYS), 
	//	options->mHwndParent,
	//	(DLGPROC)pgpDecryptionPassphraseDlgProc, (LPARAM)&gpp);

	//*(options->mPassphrasePtr)=gpp.pszPassPhrase;

	//UninitRandomKeyHook(hhookMouse,hhookKeyboard);

	return(err);
}

// Signer combo box
PGPError
pgpSigningPassphraseDialogPlatform(
	PGPContextRef						context,
	CPGPSigningPassphraseDialogOptions 	*options)
{
	PGPError err = 0;
	GPP	gpp;

	memset(&gpp,0x00,sizeof(GPP));
	gpp.context=context;
	gpp.options=options;

	/*
	 * XXX
	 * Currently expects to display only *ONE* key.
	 * The caller must set the keyset in the options to consist
	 * of the signing key that the caller wants this routine
	 * to verify
	 */
	
	// Initialize stuff
	if (!InitSigningKeyComboBox (options)) {
		return kPGPError_UserAbort; // kPGPError_Win32_NoSecret_Key
	}
	
	if (options->mPrompt)
		puts(options->mPrompt);
	else
		printf("Need a pass phrase to use this key\n");
	
	// Need to ask and get Passphrase
	

	// Ask for passphrase
	// put into CBreak mode for no echo
	// Need to get the size of the passphrase
	// Get first key out of keyset. Keyset is supposed to
	// only have one key XXX
	
	PGPKeyListRef	KeyList;
	PGPKeyIterRef	KeyIter;
	PGPKeyDBObjRef	key;
	PGPOrderKeySet (options->mKeySet,kPGPKeyOrdering_Validity,FALSE,&KeyList);
	PGPNewKeyIter (KeyList, &KeyIter);
	PGPKeyIterNextKeyDBObj (KeyIter, kPGPKeyDBObjType_Key, &key);

	while (1) {
		FreePassphrases(&gpp);
		gpp.pszPassPhrase = (char *)secAlloc (gpp.context, 500); // XXX 500
		if (gpp.pszPassPhrase) {
			PGPBoolean  bShared;
			
			/* PGPInt32 len = pgpTtyGetPass(stdout, gpp.pszPassPhrase, 500); */
			PGPInt32 len = 0;
			if (len < 0) {
				err = kPGPError_UserAbort;
				goto end;
			}
			
			*(options->mPassphrasePtr) = gpp.pszPassPhrase;
			
			// Check Shared status
			err = PGPGetKeyDBObjBooleanProperty(key, kPGPKeyProperty_IsSecretShared,
				&bShared);
			
			if (IsntPGPError(err) && bShared) {
				// So, they want to do a shared key
				*(options->mPassphraseKeyPtr) = key;
				err = kPGPError_KeyUnusableForSignature;
				goto end;
			}
			
			if (PassphraseLengthAndQualityOK(options,gpp.pszPassPhrase)) {
				if (!options->mVerifyPassphrase) {
					err = kPGPError_NoErr;
					goto end;
				}
				
				err = ValidateSigningPhrase(&gpp,gpp.pszPassPhrase,key);
				
				if (IsntPGPError(err)) {
					err = kPGPError_NoErr;
					goto end;
				}
			}
			
			// Bad passphrase/quality
			err = kPGPError_BadPassphrase;
		}
	}
	
	// Couldn't allocate passphrases
	err = kPGPError_OutOfMemory;
	
	
 end:
	PGPFreeKeyIter(KeyIter);
	PGPFreeKeyList(KeyList);
	ClearPassphrases(&gpp);
	if (err != kPGPError_NoErr)
		FreePassphrases(&gpp);
	return(err);
}

// Double edit window
	PGPError
pgpConfirmationPassphraseDialogPlatform(
	PGPContextRef								context,
	CPGPConfirmationPassphraseDialogOptions 	*options)
{
	PGPError		err				= kPGPError_NoErr;
	GtkWidget *		wDialog			= NULL;
	gint			retval			= 0;
	GtkWidget *		wSpaceEdit1		= NULL;
	GtkWidget *		wSpaceEdit2		= NULL;
	PGPUInt32		uPassphraseLen1	= 0;
	PGPUInt32		uPassphraseLen2	= 0;
	gchar *			pszPassphrase1	= NULL;
	gchar *			pszPassphrase2	= NULL;
	PGPBoolean		bFinished		= FALSE;
	GPP				gpp;

	memset(&gpp,0x00,sizeof(GPP));
	gpp.context=context;
	gpp.options=options;

	wDialog = create_PGPConfirmationPassphraseDialog();

	wSpaceEdit1 = lookup_widget( wDialog, "se_passphrase" );
	wSpaceEdit2 = lookup_widget( wDialog, "se_confirm_passphrase" );

	do
	{
		SEReset( wSpaceEdit1 );
		SEReset( wSpaceEdit2 );

		retval = gnome_dialog_run( GNOME_DIALOG( wDialog ) );
		/* 0 = OK, 1 = Cancel */
		if( retval == 0 )
		{
			uPassphraseLen1 = SEGetTextLength( wSpaceEdit1 );
			uPassphraseLen2 = SEGetTextLength( wSpaceEdit2 );

			if( IsntNull( pszPassphrase1 ) )
				secFree( pszPassphrase1 );
			if( IsntNull( pszPassphrase2 ) )
				secFree( pszPassphrase2 );

			pszPassphrase1 = (gchar *) secAlloc( context, uPassphraseLen1 + 1 );
			pszPassphrase2 = (gchar *) secAlloc( context, uPassphraseLen2 + 1 );

			SEGetText( wSpaceEdit1, pszPassphrase1 );
			SEGetText( wSpaceEdit2, pszPassphrase2 );

			/* Check passphrases match */
			if( !strcmp( pszPassphrase1, pszPassphrase2 ) )
				bFinished = TRUE;

			/* Check passphrase length */
		}
	}
	while( !bFinished && ( retval == 0 ) );

	if( IsntNull( pszPassphrase1 ) )
	{
		*(options->mPassphrasePtr) = (char *) secAlloc( context, strlen( pszPassphrase1 ) + 1 );
		strcpy( *(options->mPassphrasePtr), pszPassphrase1 );
	}
	else
		*(options->mPassphrasePtr) = NULL;

	if( GNOME_IS_DIALOG( wDialog ) )
		gnome_dialog_close( GNOME_DIALOG( wDialog ) );
	if( IsntNull( pszPassphrase1 ) )
		secFree( pszPassphrase1 );
	if( IsntNull( pszPassphrase2 ) )
		secFree( pszPassphrase2 );

	return(err);
}

	PGPError
pgpKeyPassphraseDialogPlatform(
	PGPContextRef					context,
	CPGPKeyPassphraseDialogOptions 	*options)
{
	PGPError err;
	PGPBoolean bShared,bNeedsPhrase;
	GPP	gpp;

	memset(&gpp,0x00,sizeof(GPP));
	gpp.context=context;
	gpp.options=options;

	err=PGPGetKeyDBObjBooleanProperty( options->mDefaultKey,
		kPGPKeyProperty_IsSecretShared, &bShared);
	
	if(IsntPGPError(err) && bShared)
	{
		// So, they want to do a shared key....
		return(kPGPError_KeyUnusableForSignature);
	}

	err=PGPGetKeyDBObjBooleanProperty (options->mDefaultKey,
		kPGPKeyProperty_NeedsPassphrase, &bNeedsPhrase);

	if(IsntPGPError(err) && !bNeedsPhrase)
	{
		*(options->mPassphrasePtr)=(char *)secAlloc (context, 1);
		if(*(options->mPassphrasePtr)==0)
			return(kPGPError_OutOfMemory);

		// Doesn't need a passphrase
		strcpy(*(options->mPassphrasePtr),"");
		return(kPGPError_NoErr);
	}

//	InitRandomKeyHook(&hhookMouse,&hhookKeyboard);

	//err = DialogBoxParam (g_hInst, 
	//	MAKEINTRESOURCE (IDD_KEYPASSPHRASE), 
	//	options->mHwndParent,
	//	(DLGPROC)pgpKeyPassphraseDlgProc, (LPARAM)&gpp);
//
//	*(options->mPassphrasePtr)=gpp.pszPassPhrase;
//
//	UninitRandomKeyHook(hhookMouse,hhookKeyboard);

	return(err);
}

